#!/usr/bin/env bash
set -euo pipefail

# Configure these paths for your repo layout
CONTROL_PATHS=${CONTROL_PATHS:-"src control engine"}
RESULTS_JSON=${RESULTS_JSON:-"results.json"}

echo "== Governance lints =="

# 1) No curve/pivot/logistic multipliers in control paths
for d in $CONTROL_PATHS; do
  if [ -d "$d" ]; then
    if grep -RInE 'g\(D\)|pivot|logistic' "$d"; then
      echo "[FAIL] Found curve/pivot/logistic tokens in control code."
      exit 1
    fi
  fi
done
echo "[OK] No curve/pivot/logistic tokens detected in control code."

# 2) L-roles only on arrows (basic heuristic: forbid 'L1 level' etc.)
if grep -RInE 'L[123]\s*level' $CONTROL_PATHS 2>/dev/null; then
  echo "[FAIL] Found L1/L2/L3 used as levels."
  exit 1
fi
echo "[OK] No misuse of L-roles as levels."

# 3) If UGM appears, require the phrase 'UGM is a +1 proxy'
if grep -RIn 'UGM' . 2>/dev/null; then
  if ! grep -RIn 'UGM is a +1 proxy' . 2>/dev/null; then
    echo "[FAIL] UGM mentioned but required phrase missing."
    exit 1
  fi
fi
echo "[OK] UGM phrase policy satisfied (or UGM not used)."

# 4) Results JSON present (optional check for CI harness)
if [ -f "$RESULTS_JSON" ]; then
  echo "[OK] Results JSON present: $RESULTS_JSON"
else
  echo "[WARN] RESULTS_JSON not found ($RESULTS_JSON)."
fi

echo "All lints passed."
